#include "light.h"
#include "primitives.h"
#include "color.h"
#include "scene.h"
#include "reflection.h"

Light::~Light() {
}
Spectrum Light::dE(const Scene *scene,
	const DifferentialGeometry &dg,	const Vector &w) const {
	return Spectrum(0.);
}
Spectrum Light::dE(const Scene *scene,
	const DifferentialGeometry &dg,	Vector *w) const {
	*w = Vector(0,0,0);
	return Spectrum(0.);
}
bool Light::visible(const Scene *scene, const Point &x1,
	const Point &x2) const {
	if (!CastsShadows) return 1.;
	Ray r(x1, x2-x1, 1e-3, .999);
	return scene->IntersectP(r) ? false : true;
}
float Light::unoccluded(const Scene *scene, const Point &x,
		const Vector &w) const {
	if (!CastsShadows) return 1.;
	Ray r(x, w, 1e-3);
	return scene->IntersectP(r) ? 0. : 1.;
}
Spectrum Light::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *weight) const {
	*weight = 1;
	return dE(scene, dg, wo);
}
PointLight::PointLight(bool shadows, const Transform &world2light,
	const Spectrum &power, const Point &Plight)
	: Light(shadows, world2light, power) {
	lightPos = LightToWorld(Plight);
}
Spectrum PointLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	Vector *w) const {
	*w = lightPos - dg.P;
	w->Normalize();
	if (visible(scene, dg.P, lightPos))
		return I(*w) * fabs(Dot(*w, dg.N)) /
			DistanceSquared(lightPos, dg.P);
	return Spectrum(0.);
}
InfinitePointLight::InfinitePointLight(bool shadows,
	const Transform &world2light, const Spectrum &power,
	const Vector &dir)
	: Light(shadows, world2light, power) {
	lightDir = LightToWorld(dir);
}
Spectrum InfinitePointLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	Vector *w) const {
	*w = - lightDir;
	w->Normalize();
	return unoccluded(scene, dg.P, *w) *
		Power * fabs(Dot(*w, dg.N));
}
AreaLight::AreaLight(bool shadows, const Transform &world2light,
	const Spectrum &power, Shape *s)
	: Light(shadows, world2light, power) {
	shape = s;
	Area = shape->area();
}
Spectrum AreaLight::dE(const Scene *scene,
		const DifferentialGeometry &dg,	const Vector &w) const {
	DifferentialGeometry hit;
	Ray ray(dg.P, w);
	if (shape->Intersect(ray, &hit) &&
		visible( scene, dg.P, hit.P ))
	    return L(hit.P, -w);
	return 0.;
}
Spectrum AreaLight::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *wt) const {
	shape->sample(dg, wo);
	*wt = shape->weight(dg, *wo);
	return dE(scene, dg, *wo);
}
Spectrum InfiniteAreaLight::dE(const Scene *scene,
	 	const DifferentialGeometry &dg,	const Vector &w) const {
	return unoccluded(scene, dg.P, w) * Power *
		max(0.f, Dot(w, dg.N));
}
Spectrum InfiniteAreaLight::sample_dE(const Scene *scene,
		const DifferentialGeometry &dg, Vector *wo,
		Float *weight) {
	Float u1 = RandomFloat(), u2 = RandomFloat();
	Float r = sqrtf(u2);
	Float theta = 2. * M_PI * u1;
	Float x = r * cos(theta);
	Float y = r * sin(theta);
	Float z = sqrtf(1. - x*x - y*y);
	if (RandomFloat() < .5) z *= -1;
	*wo = Vector(x, y, z);
	*wo = Vector(dg.S.x * wo->x + dg.T.x * wo->y + dg.N.x * wo->z,
		     dg.S.y * wo->x + dg.T.y * wo->y + dg.N.y * wo->z,
		     dg.S.z * wo->x + dg.T.z * wo->y + dg.N.z * wo->z);
	*weight = 2.*M_PI / fabs(wo->z);
	return dE(scene, dg, *wo);
}
AmbientLight::AmbientLight(const Spectrum &p)
	: Light(false, Transform(), p) {
}
Spectrum AmbientLight::dE(const Scene *,
		const DifferentialGeometry &, const Vector &) const {
	return Power;
}

/*void Light::EmitPhotons(Scene *scene)
{
  int counter = 0;
  for (int i = 0 ; i < nPhotons ; i++ ) {
    Float u1 = RandomFloat(), u2 = RandomFloat();
    Float x = 2*cos(2*M_PI*u2)*sqrtf(u1*(1-u1));
    Float y = 2*sin(2*M_PI*u2)*sqrtf(u1*(1-u1));
    Float z = (1 - 2 * u1);
    Ray r = Ray(Point(0.,3.,0.),Vector(x,y,z));
    //r.mint = 0.0f;
	//r.maxt = 1e8f;
//    r = LightToWorld(r);
    Surf s;
    if (scene->Intersect(r,&s)) {
      if (/*absorbed 1) {
	Float rgb[3];
	Power.ConvertToRGB(rgb);
	scene->pmap->Insert(s.dgGeom.P,rgb,r.D.Hat());
	counter++;
     }
    }
  }
  scene->pmap->ScalePow(1.0/10000.0f);
  printf("\nPhoton Hits: %d out of %d",counter,nPhotons);
}*/

void Light::EmitPhotons(Scene *scene)
{
  int counter = 0;
  bool alive;
  int depth;
  PointLight* pl = (PointLight*) this;
//  float avx=0,avy=0,avz=0;
  for (int i = 0 ; i < nPhotons ; i++ ) {
	  Float x,y,z,u1,u2;
	while(1) {
    	/*Float u1 = RandomFloat(), u2 = RandomFloat();
    	Float x = 2*cos(2*M_PI*u2)*sqrtf(u1*(1-u1));
    	Float y = 2*sin(2*M_PI*u2)*sqrtf(u1*(1-u1));
    	Float z = (1 - 2 * u1);*/

    	do {
			x = (RandomFloat() - 0.5)*2.0;
			y = (RandomFloat() - 0.5)*2.0;
			z = (RandomFloat() - 0.5)*2.0;
		//	z = -0.99;
		} while (x*x + y*y + z*z > 1 || x>-0.65 || x<-0.8  || z>-0.6 || z<-0.8 || y>0.1 ||y<-0.1);

		Ray r = Ray(pl->lightPos,Vector(x,y,z));
    	//Ray r = Ray(Point(0,0,0),Vector(x,y,z));
    	//r = LightToWorld(r);
		r.D.x = x;
		r.D.y = y;
		r.D.z = z;
		alive = true;
		depth = 0;
		//printf("%f %f %f\n",r.O.x,r.O.y,r.O.z);
    	Surf s;
    	if (scene->IntersectP(r)) {
		   	counter+= DoPhotonCalc(scene,r,Power,0,false);
		  	break;
		}
	}
  }
//  avx/=(float)counter;	avy/=(float)counter;	avz/=(float)counter;
//  printf("\naverage %f %f %f",avx,avy,avz);
  scene->pmap->ScalePow(0.075f/counter);
  scene->cmap->ScalePow(0.015f/counter);
  scene->imap->ScalePow(0.015f/counter);
  scene->envmap->ScalePow(0.00025f/counter);
  printf("\nPhoton Hits: %d out of %d",counter,nPhotons);
}

int Light::DoPhotonCalc(Scene *scene,Ray r,Spectrum c,int depth,bool isCaustic) {
  	int counter = 0;
  	PointLight* pl = (PointLight*) this;
  	Surf s;
  	if (depth>=5) return 0;
  	if (scene->Intersect(r,&s)) {
		Float rgb[3];
		c.ConvertToRGB(rgb);
	//	printf("\nORIGINAL %f %f %f",rgb[0],rgb[1],rgb[2]);
		if(isCaustic) {
			scene->cmap->Insert(s.dgGeom.P,rgb,-r.D.Hat());
		} else if (depth == 0) {
			scene->pmap->Insert(s.dgGeom.P,rgb,-r.D.Hat());
		} else {
			scene->imap->Insert(s.dgGeom.P,rgb,-r.D.Hat());
		}
		counter++;
		Float len = (s.dgGeom.P-r.O).Length();

		#ifndef VOLUMETRICS
		printf("\nNO VOLUMETRICS");
		len=-1;
		#endif
		for (int j = 0;j<(depth==0?0.5:0.2)*len;j++) {
			float u = RandomFloat();
			float t = (s.dgGeom.P.x - r.O.x)/r.D.x;
		//	u=u*u;
			t*=u;
			scene->envmap->Insert(r(t),rgb,-r.D.Hat());
		//	counter++;
		}
		Vector wi;
		BSDF *bsdf = s.getBSDF();
		for (int i = 0; i < bsdf->NumSpecular(); ++i) {
			Spectrum surfColor(0.);
			surfColor+=bsdf->f_delta(i, -r.D, &wi);
			counter+=DoPhotonCalc(scene,Ray(s.dgGeom.P,wi),c*surfColor,depth+1,true);
		}
		//if (RandomFloat() < 0.25) return counter;
		bsdf->sample_f(-r.D, &wi);
		Ray r2(s.dgGeom.P,wi);
		counter+=DoPhotonCalc(scene,r2,c,depth+1,false);
		return counter;
	}
	return 0;
}