/* CS348C PROGRAMMING ASSIGNMENT 2: USER INTERFACE MODULE.

Copyright (c) 1995 The Board of Trustees of The Leland Stanford Junior
University. All rights reserved.

Permission to use, copy, modify and distribute this software for any
purpose is hereby granted without fee, provided that the above
copyright notice and this permission notice appear in all copies of
this software and that you do not sell the software.  Commercial
licensing is available by contacting the author.

THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT WARRANTY OF ANY KIND,
EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.

Author:
   Apostolos Lerios
   Computer Science Department
   Stanford University
   U.S.A.
   http://graphics.stanford.edu/~tolis/ */


#include "scurvy.h"
#include "xsupport.h"
#include <stream.h>
#include <Xm/Frame.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/manips/SoHandleBoxManip.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>


/* TYPES. */

typedef SoSeparator *SoSeparatorP;


/* FORWARD DECLARATION. */

static void Stop(void);


/* GLOBALS. */

struct Global {
  SoXtExaminerViewer *Viewer;  // 3D scene viewer.
  SoHandleBoxManip *HandleBox; // de Boor point marker manipulator.
  SoPath *ActiveTransform;     // Path to selected transform node.

  char Buffer[1024]; // I/O buffer.

  int Degree; // Degree of spline.

  int KnotCount;     // Number of spline knots.
  double *Knot;      // Spline knots.
  int *Multiplicity; // Multiplicities of interior knots.

  int PointCount;         // Number of deBoor points.
  Point *deBoor;          // Positions of deBoor points.
  SoSeparatorP *Marker;   // de Boor point markers.
  SoCoordinate3 *Polygon; // Control polygon.

  Point *Interpolated;   // Interpolated points used to find spline point.
  int SplinePointIndex;  // Index of spline point in previous array.

  int PieceCount;        // Number of pieces per segment in linear
                         // approximation of spline.
  SoSeparatorP *Segment; // Spline segments.
  int HighlightIndex[2]; // The indexes of the first and last highlighted
                         // spline segments.

  double FrameStep;     // Parameter value change between frames.
  double LastFrame;     // Parameter value at next animation frame.
  SoTimerSensor *Timer; // Timer for frame synchronization.
  int Frozen;           // Flag: is animation frozen?

  SoSeparator *ParticleG; // Animation marker node group.
  SoSphere *Particle;     // The animation marker.
  int ShowParticle;       // Flag: is above marker visible?

  SoSeparator *InterpolationG; // Node group of polar interpolation lines.
  SoLineSet *Interpolation;    // The lines of polar interpolation.
  int ShowInterpolation;       // Flag: are the above lines visible?

  int MainLoopStarted; // Flag: has the application displayed its windows?

  Global(void)
  : ActiveTransform(0), PieceCount(100),
  FrameStep(0.01), Timer(0), ShowParticle(1), ShowInterpolation(1),
  MainLoopStarted(0) {}
};

Global G;


/* UTILITIES. */

/* Skips comment lines, i.e. empty lines or lines starting with '#',
   in the input stream Input. Returns 0 if the stream becomes exhausted
   while reading comment lines; otherwise, 1 is returned and G.Buffer
   contains the first non-comment line in the input. */

static int SkipComments(ifstream &Input) {

  while (1) {
    Input.getline(G.Buffer,1024,'\n');
    if (!Input)
      return 0;
    if (G.Buffer[0]=='#' || G.Buffer[0]=='\0')
      continue;
    else      
      return 1;
  }
}

/* Sets the colors of the spline segments. */

static void SetSegmentColors(void) {

  if (G.MainLoopStarted)
    G.Viewer->setAutoRedraw(FALSE);
  for (int i=0;i<G.KnotCount-1;i++) {
    SoMFColor &Color=((SoMaterial *)(G.Segment[i]->getChild(0)))->diffuseColor;
    if (G.ActiveTransform &&
	((i>=G.HighlightIndex[0]) && (i<=G.HighlightIndex[1])))
      Color.setHSVValue((i%2)?0.35:0.18,1.0,1.0);
    else
      Color.setHSVValue((i%2)?0.54:0.1,1.0,1.0);
  }
  if (G.MainLoopStarted) {
    G.Viewer->setAutoRedraw(TRUE);
    G.Viewer->render();
  }
}


/* ELEMENTARY CALLBACKS. */

/* Terminates program execution. */

static void Quit(void) {

  Cleanup();
  exit(0);
}

/* Saves the spline parameters in the file named FileName. */

static void Save(char *FileName) {

  ofstream Output(FileName);
  if (!Output) {
    cerr << "Could not open parameter file (" << FileName <<
      ") for output." << endl;
    return;
  }

  Output << "# Spline degree." << endl;
  Output << endl;
  Output << G.Degree << endl;
  Output << endl;
  Output << "# Knot count." << endl;
  Output << endl;
  Output << G.KnotCount << endl;
  Output << endl;
  Output << "# Knots." << endl;
  Output << endl;
  for (int i=0;i<G.KnotCount;i++)
    Output << G.Knot[i] << endl;
  Output << endl;
  Output << "# Interior knot multiplicities." << endl;
  Output << endl;
  for (i=0;i<G.KnotCount-2;i++)
    Output << G.Multiplicity[i] << endl;
  Output << endl;
  Output << "# de Boor points." << endl;
  Output << endl;
  for (i=0;i<G.PointCount;i++)
    Output << G.deBoor[i][0] << "\t" << G.deBoor[i][1] << "\t" <<
      G.deBoor[i][2] << endl;
}


/* SELECTION CALLBACKS. */

/* Attaches the manipulator to the selected de Boor point marker, and
   highlights the spline segments affected by the selected point. Path
   leads to the selected node. */

static void SelectionCB(void *,
			SoPath *Path) {

  G.ActiveTransform=Path->copy(0,Path->getLength()-1);
  G.ActiveTransform->push(0);
  G.ActiveTransform->ref();

  G.HandleBox=new(SoHandleBoxManip);
  G.HandleBox->replaceNode(G.ActiveTransform);

  GetPointInfluence(Path->getIndex(Path->getLength()-2),
		    &G.HighlightIndex[0],&G.HighlightIndex[1]);
  SetSegmentColors();
}

/* Detaches the manipulator from the selected de Boor point marker,
   and removes spline segment highlights. */

static void DeselectionCB(void *,
 			  SoPath *) {

  if (G.ActiveTransform) {
    G.HandleBox->replaceManip(G.ActiveTransform,new(SoTransform));

    G.ActiveTransform->unref();
    G.ActiveTransform=0;

    SetSegmentColors();
  }
}


/* ANIMATION CALLBACKS. */

/* Moves the animation marker to its next position along the spline. */

static void Move(void) {

  // Disable screen updates.

  G.Viewer->setAutoRedraw(FALSE);

  // Find spline point.

  GetSplinePoint(G.LastFrame,1,G.Interpolated);

  // Move particle.

  if (G.ShowParticle) {
    if (G.ParticleG->getNumChildren()==2)
      G.ParticleG->addChild(G.Particle);
    ((SoTransform *)(G.ParticleG->getChild(0)))->
      translation.setValue(G.Interpolated[G.SplinePointIndex][0],
			   G.Interpolated[G.SplinePointIndex][1],
			   G.Interpolated[G.SplinePointIndex][2]);
  } else if (G.ParticleG->getNumChildren()==3)
    G.ParticleG->removeChild(2);

  // Draw polar interpolations.

  if (G.ShowInterpolation) {
    if (G.InterpolationG->getNumChildren()==3)
      G.InterpolationG->addChild(G.Interpolation);
    SoMFVec3f &Points=((SoCoordinate3 *)(G.InterpolationG->getChild(2)))->
      point;
    for (int i=0;i<G.SplinePointIndex;i++)
      Points.set1Value(i,G.Interpolated[i][0],
		       G.Interpolated[i][1],
		       G.Interpolated[i][2]);
  } else if (G.InterpolationG->getNumChildren()==4)
    G.InterpolationG->removeChild(3);

  // Update screen.

  G.Viewer->setAutoRedraw(TRUE);
  G.Viewer->render();

  // Prepare for the next frame, or stop animation if the particle
  // has moved outside the endpoints of the spline.

  G.LastFrame+=G.FrameStep;
  if (G.LastFrame<G.Knot[0] || G.LastFrame>G.Knot[G.KnotCount-1])
    Stop();
}

/* Freezes or unfreezes an on-going animation. */

static void Freeze(void) {

  if (!G.Timer)
    return;

  if (G.Frozen)
    G.Timer->schedule();
  else
    G.Timer->unschedule();
  G.Frozen=!G.Frozen;
}

/* Animates the path of a particle traveling along the spline. */

static void Animate(void) {

  // Translate request to (un)freezing if an animation is in progress.

  if (G.Timer) {
    Freeze();
    return;
  }

  // Set timer.

  G.Timer=new(SoTimerSensor)((SoSensorCB *)Move,0);
  G.Timer->schedule();
  G.Frozen=0;

  // Draw first frame.

  G.LastFrame=G.Knot[0];
  Move();
}

/* Terminates an on-going animation. */

static void Stop(void) {

  // Remove timer.

  if (!G.Timer)
    return;
  delete(G.Timer);
  G.Timer=0;

  // Update screen.

  G.Viewer->setAutoRedraw(FALSE);
  if (G.ParticleG->getNumChildren()==3)
    G.ParticleG->removeChild(2);
  if (G.InterpolationG->getNumChildren()==4)
    G.InterpolationG->removeChild(3);
  G.Viewer->setAutoRedraw(TRUE);
  G.Viewer->render();
}


/* SPLINE DRAWING CALLBACKS. */

/* Draws the spline. */

static void DrawSpline(void) {

  // Disable screen updates.

  if (G.MainLoopStarted)
    G.Viewer->setAutoRedraw(FALSE);

  // Find points of spline's linear approximation.

  for (int i=0;i<G.KnotCount-1;i++) {
    double t=G.Knot[i];
    double dt=(G.Knot[i+1]-G.Knot[i])/G.PieceCount;
    SoMFVec3f &Segment=((SoCoordinate3 *)(G.Segment[i]->getChild(1)))->point;
    for (int j=0;j<=G.PieceCount;j++) {
      GetSplinePoint(t,(j<G.PieceCount)?1:(-1),G.Interpolated);
      Segment.set1Value(j,G.Interpolated[G.SplinePointIndex][0],
			G.Interpolated[G.SplinePointIndex][1],
			G.Interpolated[G.SplinePointIndex][2]);
      if (j==G.PieceCount-1)
	t=G.Knot[i+1];
      else
	t+=dt;
    }
    Segment.setNum(G.PieceCount+1);
  }

  // Update screen.

  if (G.MainLoopStarted) {
    G.Viewer->setAutoRedraw(TRUE);
    G.Viewer->render();
  }
}

/* Moves the i+1 st de Boor point. */

static void ExtractPosition(int i) {

  const SbVec3f &Position=
    ((SoTransform *)(G.Marker[i]->getChild(0)))->translation.getValue();
  G.Polygon->point.set1Value(i,Position);
  for (int j=0;j<3;j++)
    G.deBoor[i][j]=Position[j];
  DrawSpline();
}


/* CHOICE BOX CALLBACKS. */

/* Sets the visibility of the interpolation lines during
   animation. Set indicates whether the lines are visible (1) or not
   (0). */

static void ToggleInterpolation(int Set) {

  G.ShowInterpolation=Set;
}

/* Sets the visibility of the particle during animation. Set indicates
   whether the particle is visible (1) or not (0). */

static void ToggleParticle(int Set) {

  G.ShowParticle=Set;
}


/* SLIDER CALLBACKS. */

/* Sets the piece count to NewPieceCount. */

static void NewPieceCount(float NewPieceCount,
			  void *) {

  G.PieceCount=int(NewPieceCount);
  DrawSpline();
}

/* Sets the frame step to NewFrameStep. */

static void NewFrameStep(float NewFrameStep,
                         void *) {

  G.FrameStep=NewFrameStep;
}

/* Sets the distance between knots KnotIndex and KnotIndex+1 to
   NewKnotSpacing. */

static void NewKnotSpacing(float NewKnotSpacing,
			   int KnotIndex) {

  double Difference=NewKnotSpacing-(G.Knot[KnotIndex]-G.Knot[KnotIndex-1]);
  for (int i=KnotIndex;i<G.KnotCount;i++)
    G.Knot[i]+=Difference;
  NewKnots();
  DrawSpline();
}


/* MAIN PROGRAM. */

int main(int argc,
	 char *argv[]) {

  // Copyright.

  cerr << endl;
  cerr << "  B-spline curve visualization." << endl;
  cerr << endl;
  cerr << "         Apostolos Lerios" << endl;
  cerr << "        Stanford University" << endl;
  cerr << "http://graphics.stanford.edu/~tolis/" << endl;
  cerr << endl;

  // Parse command-line.

  for (int i=1;i<argc;i++) {
    if (!strcmp(argv[i],"-help")) {
      cerr << "Usage: " << endl;
      cerr << endl;
      cerr << argv[0] << " <parameter file>" << endl;
      cerr << endl;
      cerr << "-help: shows this help screen." << endl;
      cerr << endl;
      return 1;
    }
  }
  if (argc<2) {
    cerr << argv[0] << 
      " should be followed by the name of the input parameter file." << endl;
    return 1;
  }

  // Parse parameter file.

  ifstream Input(argv[1]);
  if (!Input) {
    cerr << "Could not open parameter file (" << argv[1] << ")." << endl;
    return 1;
  }
  cerr << "*********************************************************" << endl;
  cerr << endl;
  cerr << "Initial settings:" << endl;
  cerr << endl;

  if (!SkipComments(Input) || (sscanf(G.Buffer,"%d",&G.Degree)<1)) {
    cerr << "Expecting degree of the spline in parameter file (" <<
      argv[1] << ")." << endl;
    return 1;
  }
  if (G.Degree<1) {
    cerr << "Spline degree must be at least 1." << endl;
    return 1;
  }
  cerr << "Spline degree: " << G.Degree << endl;

  if (!SkipComments(Input) || (sscanf(G.Buffer,"%d",&G.KnotCount)<1)) {
    cerr << "Expecting number of knots in parameter file (" <<
      argv[1] << ")." << endl;
    return 1;
  }
  if (G.KnotCount<2) {
    cerr << "Knot count must be at least 2." << endl;
    return 1;
  }
  cerr << "Knot count: " << G.KnotCount << endl;
  G.Knot=new(double[G.KnotCount]);
  G.Multiplicity=new(int[G.KnotCount-2]);

  for (i=0;i<G.KnotCount;i++) {
    if (!SkipComments(Input) || (sscanf(G.Buffer,"%lf",&G.Knot[i])<1)) {
      cerr << "Expecting value of knot " << i+1 <<
	" in parameter file (" << argv[1] << ")." << endl;
      return 1;
    }
    if ((i>0) && (G.Knot[i]<=G.Knot[i-1])) {
      cerr << "Knot " << i+1 <<	" is not strictly greater than knot " << i <<
	"." << endl;
      return 1;
    }
  }
  cerr << "Knots:" << endl;
  for (i=0;i<G.KnotCount;i++)
    cerr << " " << G.Knot[i];
  cerr << endl;

  for (i=0;i<G.KnotCount-2;i++) {
    if (!SkipComments(Input) ||
	(sscanf(G.Buffer,"%d",&G.Multiplicity[i])<1)) {
      cerr << "Expecting mutliplicity for knot " << i+2 <<
	" in parameter file (" << argv[1] << ")." << endl;
      return 1;
    }
    if ((G.Multiplicity[i]<1) || (G.Multiplicity[i]>G.Degree+1)) {
      cerr << "Multiplicity of knot " << i+2 << " should be between 1 and " <<
	G.Degree+1 << " for a spline of degree " << G.Degree << "." << endl;
      return 1;
    }
  }
  cerr << "Interior knot multiplicities:" << endl;
  G.PointCount=G.Degree+1;
  for (i=0;i<G.KnotCount-2;i++) {
    cerr << " " << G.Multiplicity[i];
    G.PointCount+=G.Multiplicity[i];
  }
  cerr << endl;
  G.deBoor=new(Point[G.PointCount]);

  for (i=0;i<G.PointCount;i++) {
    if (!SkipComments(Input) ||
	(sscanf(G.Buffer,"%lf %lf %lf",
		&G.deBoor[i][0],&G.deBoor[i][1],&G.deBoor[i][2])<3)) {
      cerr << "Expecting de Boor point " << i+1 << " in parameter file (" <<
	argv[1] << ")." << endl;
      return 1;
    }
  }
  cerr << "de Boor points:" << endl;
  for (i=0;i<G.PointCount;i++)
    cerr << " " << G.deBoor[i][0] << " " << G.deBoor[i][1] << " " <<
      G.deBoor[i][2] << endl;

  cerr << endl;
  cerr << "*********************************************************" << endl;
  cerr << endl;

  // Initialize Inventor and Xt.

  Widget ApplicationWindow=SoXt::init(argv[0]);
  if (!ApplicationWindow) {
    cerr << "Could not create application window." << endl;
    return 1;
  }

  // Create main window.

  Widget Main=XtVaCreateWidget("",xmFormWidgetClass,ApplicationWindow,
                               NULL);
  Widget Controls=XtVaCreateWidget("",xmRowColumnWidgetClass,Main,
				   XmNtopAttachment,XmATTACH_FORM,
				   XmNleftAttachment,XmATTACH_FORM,
				   XmNrightAttachment,XmATTACH_FORM,
				   NULL);

  // Buttons (general).

  Widget Row=XtVaCreateWidget("",xmRowColumnWidgetClass,Controls,
                              XmNorientation,XmHORIZONTAL,
                              NULL);
  new(PushButton)(Row,"Quit",PushButtonCallback(Quit));
  new(DialogButton)(Row,"Save","Save spline parameters in",
		    DialogButtonCallback(Save));
  XtManageChild(Row);

  // Buttons (animation).

  Row=XtVaCreateWidget("",xmRowColumnWidgetClass,Controls,
		       XmNorientation,XmHORIZONTAL,
		       NULL);
  new(PushButton)(Row,"Animate",PushButtonCallback(Animate));
  new(PushButton)(Row,"(Un)Freeze",PushButtonCallback(Freeze));
  new(PushButton)(Row,"Stop",PushButtonCallback(Stop));
  XtManageChild(Row);

  // Choice boxes.

  ChoiceButtonSet *Buttons=new(ChoiceButtonSet)(Controls,
						"Animation:",0);
  Buttons->AddButton("Particle",1,ToggleParticle);
  Buttons->AddButton("Interpolations",1,ToggleInterpolation);

  // Sliders (pieces per segment and animation step).
  
  Row=XtVaCreateWidget("",xmRowColumnWidgetClass,Controls,
		       XmNorientation,XmHORIZONTAL,
		       NULL);
  new(Slider)(Row,"Pieces",10,1000,G.PieceCount,0,
	      SliderCallback(NewPieceCount));
  new(Slider)(Row,"Frame Step",-100,100,int(G.FrameStep*1000.0),3,
              SliderCallback(NewFrameStep));
  XtManageChild(Row);

  // Sliders (knot spacing).

  double MaxSpacing=G.Knot[1]-G.Knot[0];
  for (i=1;i<G.KnotCount-1;i++) {
    double Spacing=G.Knot[i+1]-G.Knot[i];
    if (Spacing>MaxSpacing)
      MaxSpacing=Spacing;
  }
  for (i=0;i<G.KnotCount-1;i++) {
    if (i%5==0) 
      Row=XtVaCreateManagedWidget("",xmRowColumnWidgetClass,Controls,
				  XmNorientation,XmHORIZONTAL,
				  NULL);
    sprintf(G.Buffer,"Knot %d to %d",i+1,i+2);
    new(Slider)(Row,G.Buffer,1,int(MaxSpacing*300),
		int((G.Knot[i+1]-G.Knot[i])*100),2,
		SliderCallback(NewKnotSpacing),(void *)(i+1));
  }

  XtManageChild(Controls);

  // Scene graph (de Boor points).

  SoSeparator *Scene=new(SoSeparator);

  SoSelection *Selection=new(SoSelection);
  Scene->addChild(Selection);
  Selection->addSelectionCallback((SoSelectionPathCB *)SelectionCB,0);
  Selection->addDeselectionCallback((SoSelectionPathCB *)DeselectionCB,0);

  G.Marker=new(SoSeparatorP[G.PointCount]);
  for (i=0;i<G.PointCount;i++) {
    G.Marker[i]=new(SoSeparator);

    // Marker movement callback.

    new(SoNodeSensor)((SoSensorCB *)ExtractPosition,(void *)i)->
      attach(G.Marker[i]);

    // Position of de Boor point.

    SoTransform *Transform=new(SoTransform);
    Transform->translation.setValue(G.deBoor[i][0],
				    G.deBoor[i][1],
				    G.deBoor[i][2]);
    G.Marker[i]->addChild(Transform);

    // Color of de Boor point marker.

    SoMaterial *Material=new(SoMaterial);
    Material->diffuseColor.setHSVValue(0.66+0.33*i/(G.PointCount-1.0),1.0,1.0);
    G.Marker[i]->addChild(Material);

    // Shape of marker.

    G.Marker[i]->addChild(new(SoSphere));

    Selection->addChild(G.Marker[i]);
  }

  Scene->addChild(Selection);

  // Scene graph (control polygon).

  SoSeparator *Polygon=new(SoSeparator);

  SoLightModel *LightModel=new(SoLightModel);
  LightModel->model=SoLightModel::BASE_COLOR;
  Polygon->addChild(LightModel);

  G.Polygon=new(SoCoordinate3);
  for (i=0;i<G.PointCount;i++)
    G.Polygon->point.set1Value(i,G.deBoor[i][0],G.deBoor[i][1],G.deBoor[i][2]);
  Polygon->addChild(G.Polygon);

  Polygon->addChild(new(SoLineSet));

  Scene->addChild(Polygon);

  // Scene graph (spline visualization).

  SoSeparator *Spline=new(SoSeparator);
  
  LightModel=new(SoLightModel);
  LightModel->model=SoLightModel::BASE_COLOR;
  Spline->addChild(LightModel);

  G.Segment=new(SoSeparatorP[G.KnotCount-1]);
  for (i=0;i<G.KnotCount-1;i++) {
    G.Segment[i]=new(SoSeparator);
    G.Segment[i]->addChild(new(SoMaterial));
    G.Segment[i]->addChild(new(SoCoordinate3));
    G.Segment[i]->addChild(new(SoLineSet));
    Spline->addChild(G.Segment[i]);
  }
  SetSegmentColors();

  Scene->addChild(Spline);

  // Scene graph (animation).

  G.ParticleG=new(SoSeparator);

  G.ParticleG->addChild(new(SoTransform));

  SoMaterial *Material=new(SoMaterial);
  Material->diffuseColor.setHSVValue(0.35,1.0,1.0);
  G.ParticleG->addChild(Material);

  Scene->addChild(G.ParticleG);

  G.Particle=new(SoSphere);
  G.Particle->radius=0.5;
  G.Particle->ref();

  // Scene graph (polar interpolations).

  G.InterpolationG=new(SoSeparator);

  LightModel=new(SoLightModel);
  LightModel->model=SoLightModel::BASE_COLOR;
  G.InterpolationG->addChild(LightModel);

  Material=new(SoMaterial);
  Material->diffuseColor.setHSVValue(0.35,1.0,1.0);
  G.InterpolationG->addChild(Material);

  G.InterpolationG->addChild(new(SoCoordinate3));

  Scene->addChild(G.InterpolationG);

  G.Interpolation=new(SoLineSet);
  int j;
  for (i=0,j=G.Degree+1;i<G.Degree;i++,j--)
    G.Interpolation->numVertices.set1Value(i,j);
  G.Interpolation->ref();

  // Initialize spline drawing module.

  InitSystem(G.Degree,G.KnotCount,G.Knot,G.Multiplicity,G.deBoor);
  NewKnots();
  G.SplinePointIndex=(G.Degree+1)*(G.Degree+2)/2-1;
  G.Interpolated=new(Point[G.SplinePointIndex+1]);
  DrawSpline();

  // Viewer.

  Widget Frame=XtVaCreateWidget("",xmFrameWidgetClass,Main,
				XmNshadowType,XmSHADOW_ETCHED_IN,
				XmNtopAttachment,XmATTACH_WIDGET,
				XmNtopWidget,Controls,
				XmNleftAttachment,XmATTACH_FORM,
				XmNrightAttachment,XmATTACH_FORM,
				XmNbottomAttachment,XmATTACH_FORM,
				NULL);
  G.Viewer=new(SoXtExaminerViewer)(Frame);
  G.Viewer->setSceneGraph(Scene);
  G.Viewer->setSize(SbVec2s(600,600));
  G.Viewer->show();

  // Pop up window and yield control to Inventor.

  XtManageChild(Frame);
  XtManageChild(Main);
  SoXt::show(ApplicationWindow);
  G.MainLoopStarted=1;
  SoXt::mainLoop();
  return 0;
}
